from sqlalchemy import Column, Integer, String, Boolean, JSON, ForeignKey, DECIMAL, Text, TIMESTAMP
from sqlalchemy.ext.declarative import declarative_base
from fastapi import APIRouter, HTTPException, Depends
from sqlalchemy.orm import Session
# Adjust the following imports to your project structure
from .schemas import OrderBlanketIn, OrderBlanketOut  # adjust import as needed
from ..database import get_db  # adjust import as needed

Base = declarative_base()

class OrderBlanket(Base):
    __tablename__ = 'order_blankets'

    id = Column(Integer, primary_key=True)
    order_id = Column(Integer, ForeignKey('orders.id'), nullable=False)
    type = Column(String(50))  # Blanket category (Turnout, Stable, etc.)
    weight = Column(String(50))  # Blanket weight (winter, mid_weight, etc.)
    brand = Column(String(100))
    quantity = Column(Integer)
    colors = Column(JSON)  # JSON array of color codes/names
    is_plaid = Column(Boolean, default=False)
    has_water_treatment = Column(Boolean, default=False)
    has_hood = Column(Boolean, default=False)
    has_heavy_soil = Column(Boolean, default=False)
    has_heavy_hair = Column(Boolean, default=False)
    has_repairs = Column(Boolean, default=False)
    repair_minutes = Column(Integer)
    repair_notes = Column(Text)
    blanket_quality_colour = Column(String(20))
    base_price = Column(DECIMAL(10,2))
    is_double_weight = Column(Boolean, default=False)
    dog_type = Column(String(50))
    foal_mini_type = Column(String(50))
    blanket_style = Column(String(100))
    note_category = Column(String(100))
    water_treatment_price = Column(DECIMAL(10,2))
    discount = Column(JSON)  # JSON for discount details
    notes = Column(Text)
    no_wash = Column(Boolean, default=False)
    pattern_notes = Column(Text)
    has_pattern = Column(Boolean, default=False)
    created_at = Column(TIMESTAMP)
    updated_at = Column(TIMESTAMP)
    blanket_type_id = Column(Integer, ForeignKey('blanket_types.id'), nullable=False)
    red_flags = Column(JSON)  # JSON for any red flag indicators (optional)
    # Add any new fields you want to support here 

router = APIRouter()

@router.post('/order_blankets/', response_model=OrderBlanketOut)
def create_blanket(blanket: OrderBlanketIn, db: Session = Depends(get_db)):
    db_blanket = OrderBlanket(**blanket.dict())
    db.add(db_blanket)
    db.commit()
    db.refresh(db_blanket)
    return db_blanket

@router.get('/order_blankets/{blanket_id}', response_model=OrderBlanketOut)
def get_blanket(blanket_id: int, db: Session = Depends(get_db)):
    blanket = db.query(OrderBlanket).get(blanket_id)
    if not blanket:
        raise HTTPException(status_code=404, detail="Blanket not found")
    return blanket

@router.put('/order_blankets/{blanket_id}', response_model=OrderBlanketOut)
def update_blanket(blanket_id: int, blanket: OrderBlanketIn, db: Session = Depends(get_db)):
    db_blanket = db.query(OrderBlanket).get(blanket_id)
    if not db_blanket:
        raise HTTPException(status_code=404, detail="Blanket not found")
    for key, value in blanket.dict().items():
        setattr(db_blanket, key, value)
    db.commit()
    db.refresh(db_blanket)
    return db_blanket 