-- Drop existing tables if they exist
DROP TABLE IF EXISTS order_add_ons;
DROP TABLE IF EXISTS order_blankets;
DROP TABLE IF EXISTS add_ons;
DROP TABLE IF EXISTS blanket_types;

-- Create blanket_types table
CREATE TABLE IF NOT EXISTS blanket_types (
  id INT AUTO_INCREMENT PRIMARY KEY,
  type VARCHAR(50) NOT NULL,
  category VARCHAR(50) NOT NULL, -- 'turnout', 'stable', 'scrim'
  weight VARCHAR(50) NOT NULL, -- 'winter', 'mid_weight', 'single_weight', 'light_weight'
  price DECIMAL(10, 2) NOT NULL,
  description TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create add_ons table
CREATE TABLE IF NOT EXISTS add_ons (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  type VARCHAR(50) NOT NULL, -- 'water_treatment', 'heavy_soil', 'heavy_hair', 'repair'
  price DECIMAL(10, 2) NOT NULL,
  description TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create order_blankets junction table
CREATE TABLE IF NOT EXISTS order_blankets (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_id INT NOT NULL,
  blanket_type_id INT NOT NULL,
  quantity INT NOT NULL DEFAULT 1,
  notes TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
  FOREIGN KEY (blanket_type_id) REFERENCES blanket_types(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create order_add_ons junction table
CREATE TABLE IF NOT EXISTS order_add_ons (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_id INT NOT NULL,
  add_on_id INT NOT NULL,
  quantity INT NOT NULL DEFAULT 1,
  notes TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
  FOREIGN KEY (add_on_id) REFERENCES add_ons(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert standard blanket types
INSERT INTO blanket_types (type, category, weight, price) VALUES
-- Turnout
('Turnout', 'turnout', 'winter', 24.50),
('Turnout', 'turnout', 'mid_weight', 22.50),
('Turnout', 'turnout', 'single_weight', 19.50),
('Hood', 'turnout', 'standard', 8.00),

-- Stable, Liner, Cooler
('Stable', 'stable', 'winter', 24.50),
('Stable', 'stable', 'mid_weight', 22.50),
('Stable', 'stable', 'single_weight', 20.50),
('Liner', 'stable', 'winter', 24.50),
('Liner', 'stable', 'mid_weight', 22.50),
('Liner', 'stable', 'single_weight', 20.50),
('Cooler', 'stable', 'winter', 24.50),
('Cooler', 'stable', 'mid_weight', 22.50),
('Cooler', 'stable', 'single_weight', 20.50),
('Hood', 'stable', 'standard', 8.00),

-- Scrim, Flysheet, Nylon
('Scrim', 'scrim', 'mid_weight', 20.50),
('Scrim', 'scrim', 'light_weight', 19.50),
('Flysheet', 'scrim', 'mid_weight', 20.50),
('Flysheet', 'scrim', 'light_weight', 19.50),
('Nylon', 'scrim', 'mid_weight', 20.50),
('Nylon', 'scrim', 'light_weight', 19.50),
('Hood', 'scrim', 'standard', 8.00);

-- Insert standard add-ons
INSERT INTO add_ons (name, type, price) VALUES
('Water Treatment', 'water_treatment', 19.00),
('Hood Water Treatment', 'water_treatment', 8.00),
('Heavy Soil', 'heavy_soil', 8.00),
('Heavy Hair', 'heavy_hair', 8.00),
('Repairs', 'repair', 36.00); -- Base price per hour, will be calculated in 5-minute increments with 15-minute minimum

-- Add indexes for better performance
CREATE INDEX idx_blanket_types_type ON blanket_types(type);
CREATE INDEX idx_blanket_types_category ON blanket_types(category);
CREATE INDEX idx_blanket_types_weight ON blanket_types(weight);
CREATE INDEX idx_add_ons_type ON add_ons(type);
CREATE INDEX idx_order_blankets_order_id ON order_blankets(order_id);
CREATE INDEX idx_order_blankets_blanket_type_id ON order_blankets(blanket_type_id);
CREATE INDEX idx_order_add_ons_order_id ON order_add_ons(order_id);
CREATE INDEX idx_order_add_ons_add_on_id ON order_add_ons(add_on_id);

-- Add composite indexes for common query patterns
CREATE INDEX idx_blanket_types_category_weight ON blanket_types(category, weight);
CREATE INDEX idx_order_blankets_order_blanket ON order_blankets(order_id, blanket_type_id);
CREATE INDEX idx_order_add_ons_order_addon ON order_add_ons(order_id, add_on_id);

-- Drop views if they exist
DROP VIEW IF EXISTS v_blanket_order_history;
DROP VIEW IF EXISTS v_addon_order_history;

-- Create view for blanket order history
CREATE VIEW v_blanket_order_history AS
SELECT 
    o.id as order_id,
    o.order_number,
    o.date as order_date,
    c.id as customer_id,
    c.name as customer_name,
    s.id as store_id,
    s.name as store_name,
    bt.type as blanket_type,
    bt.category as blanket_category,
    bt.weight as blanket_weight,
    ob.quantity,
    ob.notes
FROM orders o
JOIN customers c ON o.customer_id = c.id
JOIN stores s ON o.store_id = s.id
JOIN order_blankets ob ON o.id = ob.order_id
JOIN blanket_types bt ON ob.blanket_type_id = bt.id;

-- Create view for add-on order history
CREATE VIEW v_addon_order_history AS
SELECT 
    o.id as order_id,
    o.order_number,
    o.date as order_date,
    c.id as customer_id,
    c.name as customer_name,
    s.id as store_id,
    s.name as store_name,
    ao.name as addon_name,
    ao.type as addon_type,
    oa.quantity,
    oa.notes
FROM orders o
JOIN customers c ON o.customer_id = c.id
JOIN stores s ON o.store_id = s.id
JOIN order_add_ons oa ON o.id = oa.order_id
JOIN add_ons ao ON oa.add_on_id = ao.id; 