# Load environment variables
from dotenv import load_dotenv
load_dotenv()

from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from app.routers import orders
from app.routers import customers
from app.database import init_db
import logging

# Set up logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = FastAPI()

# Configure CORS
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],  # In production, set to your frontend's domain!
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Initialize the database at app startup
@app.on_event("startup")
async def startup_event():
    """
    Initializes the database connection when the application starts.
    """
    try:
        logger.info("Initializing database...")
        init_db()
        logger.info("Database initialization complete.")
    except Exception as e:
        logger.error(f"FATAL: Error during database initialization: {str(e)}")
        raise

# Include the API routers
app.include_router(orders.router, prefix="/api", tags=["orders"])
app.include_router(customers.router, prefix="/api", tags=["customers"])

@app.get("/")
def read_root():
    """
    A simple root endpoint to confirm the API is running.
    """
    return {"message": "Welcome to the Athletic Clean API"}
