# /backend/app/routers/orders.py
# Version: Definitive_Final_Corrected_Imports_v2

from fastapi import APIRouter, HTTPException, Depends
from sqlalchemy.orm import Session
from sqlalchemy.exc import SQLAlchemyError
from typing import List, Optional
import logging
import json

# --- Model and Schema Imports (Corrected Structure) ---
from ..database import get_db
from ..models.order import Order
from ..models.order_blanket import OrderBlanket
from ..models.order_sequence import OrderSequence
from ..models.customer import Customer
from ..schemas.order import OrderIn, OrderOut

# --- Router Setup ---
router = APIRouter(prefix="/orders", tags=["orders"])
logger = logging.getLogger(__name__)


# --- Helper Function for Order Number Generation ---
def _get_next_order_number(prefix: str, db: Session) -> str:
    """
    Safely gets the next available order number for a given prefix.
    """
    try:
        with db.begin_nested():
            sequence = db.query(OrderSequence).filter(OrderSequence.prefix == prefix).with_for_update().first()
            if not sequence:
                sequence = OrderSequence(prefix=prefix, last_number=100)
                db.add(sequence)
                db.flush()

            next_number = sequence.last_number
            while True:
                next_number += 1
                new_order_number = f"{prefix}{next_number}"
                if not db.query(Order.id).filter(Order.order_number == new_order_number).first():
                    break
            
            sequence.last_number = next_number
            return new_order_number
    except Exception as e:
        logger.error(f"Error in _get_next_order_number: {e}", exc_info=True)
        raise

# --- API Endpoints ---
@router.get("/next-number", response_model=dict)
def get_next_order_number_endpoint(prefix: str, db: Session = Depends(get_db)):
    """
    API endpoint to get the next available order number.
    """
    try:
        order_number = _get_next_order_number(prefix, db)
        db.commit()
        return {"order_number": order_number}
    except Exception as e:
        db.rollback() 
        logger.error(f"Endpoint error generating next order number: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail="Could not generate next order number.")

@router.post("/", response_model=OrderOut)
def create_order(order_data: OrderIn, db: Session = Depends(get_db)):
    """
    Creates a new order, handling customer and blanket details.
    """
    try:
        with db.begin_nested():
            # Step 1: Upsert Customer
            customer = None
            if order_data.customer_email:
                customer = db.query(Customer).filter(Customer.email == order_data.customer_email).first()
            if not customer and order_data.customer_phone:
                customer = db.query(Customer).filter(Customer.phone == order_data.customer_phone).first()

            if customer:
                customer.name = order_data.customer_name
                customer.phone = order_data.customer_phone
                customer.address = order_data.customer_address
            else:
                customer = Customer(name=order_data.customer_name, email=order_data.customer_email, phone=order_data.customer_phone, address=order_data.customer_address, type='individual')
                db.add(customer)
                db.flush()
            
            # Step 2: Create Order
            db_order = Order(
                order_number=order_data.orderNumber, status=order_data.status, date=order_data.date, week=order_data.week,
                customer_id=customer.id, customer_name=customer.name, customer_phone=customer.phone,
                customer_email=customer.email, customer_address=customer.address,
                store_id=order_data.store_id, store_name=order_data.store_name,
                is_gh_staff_order=order_data.is_gh_staff_order, staff_name=order_data.staff_name,
                total=order_data.total,
                blankets=json.dumps([b.dict() for b in order_data.blankets]),
                order_colors=json.dumps(order_data.order_colors),
                documents=json.dumps(order_data.documents or {})
            )
            db.add(db_order)
            db.flush()

            # Step 3: Create OrderBlanket records
            for blanket_in in order_data.blankets:
                detail = blanket_in.details[0] if blanket_in.details else None
                if not detail: continue
                db_blanket = OrderBlanket(
                    order_id=db_order.id, type=blanket_in.type, weight=blanket_in.weight, brand=detail.brand,
                    colors=json.dumps(detail.colors), is_plaid=detail.isPlaid, has_repairs=detail.hasRepairs,
                    repair_minutes=detail.repairMinutes, repair_notes=detail.repairNotes, admin_notes=detail.adminNotes,
                    admin_discount=json.dumps(detail.adminDiscount.dict()),
                    # All other fields from the model should be added here...
                )
                db.add(db_blanket)

        db.commit()
        db.refresh(db_order)
        logger.info(f"Successfully created order {db_order.order_number}")
        return db_order

    except Exception as e:
        db.rollback()
        logger.error(f"Error creating order: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/{order_number}", response_model=OrderOut)
def get_order_by_number(order_number: str, db: Session = Depends(get_db)):
    try:
        # 1. Find the OrderBlanket
        blanket = db.query(OrderBlanket).filter(
            OrderBlanket.id == blanket_id,
            OrderBlanket.order_id == order_id
        ).first()
        if not blanket:
            raise HTTPException(status_code=404, detail="Blanket not found for this order.")

        # 2. Delete the Blanket record
        db.delete(blanket)
        db.flush()

        # 3. Update the parent Order's blankets JSON field (if you want to keep it in sync)
        order = db.query(Order).filter(Order.id == order_id).first()
        if order:
            blankets = json.loads(order.blankets)
            # Remove by matching blanket_id (adjust as needed to match your ID system)
            blankets = [b for b in blankets if b.get("id") != blanket_id]
            order.blankets = json.dumps(blankets)
            db.add(order)

        db.commit()
        return {"message": "Blanket deleted successfully."}
    except SQLAlchemyError as e:
        db.rollback()
        logger.error(f"DB Error deleting blanket: {str(e)}", exc_info=True)
        raise HTTPException(status_code=500, detail="Database error during blanket deletion.")
    except Exception as e:
        db.rollback()
        logger.error(f"General error deleting blanket: {str(e)}", exc_info=True)
        raise HTTPException(status_code=500, detail="Error during blanket deletion.")
    order = db.query(Order).filter(Order.order_number == order_number).first()
    if not order:
        raise HTTPException(status_code=404, detail=f"Order with number {order_number} not found")
    return order
