from fastapi import APIRouter, HTTPException, Depends
from sqlalchemy.orm import Session
from ..models.order import Order, OrderStatus
from ..models.invoice import Invoice
from ..database import get_db
from sqlalchemy import Column, Integer, String, DateTime, ForeignKey, JSON, Text
from sqlalchemy.orm import relationship
from datetime import datetime
import json
from ..models.order_sequence import OrderSequence

router = APIRouter()

def calculate_order_total(order: Order) -> float:
    # TODO: Replace with actual logic
    return 100.0

@router.patch("/orders/{order_id}/validate")
def validate_order(order_id: int, db: Session = Depends(get_db)):
    order = db.query(Order).filter(Order.id == order_id).first()
    if not order:
        raise HTTPException(status_code=404, detail="Order not found")
    if order.status != OrderStatus.CREATED:
        raise HTTPException(status_code=400, detail="Order is not in a valid state")
    order.status = OrderStatus.VALIDATED
    db.commit()
    return {"status": "Order validated"}

@router.post("/orders/{order_id}/invoice")
def create_invoice(order_id: int, db: Session = Depends(get_db)):
    order = db.query(Order).filter(Order.id == order_id).first()
    if not order or order.status != OrderStatus.VALIDATED:
        raise HTTPException(status_code=400, detail="Order must be validated first")

    total = calculate_order_total(order)
    hst = 0.13 * total if not order.store.is_corporate else 0
    invoice = Invoice(order_id=order.id, total=total + hst, hst_included=not order.store.is_corporate)
    order.status = OrderStatus.INVOICED

    db.add(invoice)
    db.commit()
    return {"status": "Invoice created", "invoice_id": invoice.id}

@router.patch("/invoices/{invoice_id}/validate")
def validate_sale(invoice_id: int, db: Session = Depends(get_db)):
    invoice = db.query(Invoice).filter(Invoice.id == invoice_id).first()
    if not invoice:
        raise HTTPException(status_code=404, detail="Invoice not found")

    order = invoice.order
    if order.status != OrderStatus.INVOICED:
        raise HTTPException(status_code=400, detail="Order is not in invoiced state")

    order.status = OrderStatus.SALE_VALIDATED
    db.commit()
    return {"status": "Sale validated"}

@router.patch("/invoices/{invoice_id}/post")
def post_payment(invoice_id: int, db: Session = Depends(get_db)):
    invoice = db.query(Invoice).filter(Invoice.id == invoice_id).first()
    if not invoice:
        raise HTTPException(status_code=404, detail="Invoice not found")

    if not invoice.order or invoice.order.status != OrderStatus.SALE_VALIDATED:
        raise HTTPException(status_code=400, detail="Sale must be validated first")

    invoice.paid = True
    invoice.order.status = OrderStatus.PAID
    db.commit()
    return {"status": "Payment posted and sale completed"}

@router.get("/orders/next-number")
def get_next_order_number(prefix: str, db: Session = Depends(get_db)):
    sequence = db.query(OrderSequence).filter(OrderSequence.prefix == prefix).first()
    if not sequence:
        return {"next_number": f"{prefix}100"}
    return {"next_number": f"{prefix}{sequence.last_number + 1}"}

@router.post("/orders")
def create_order(order_data: dict, db: Session = Depends(get_db)):
    prefix = order_data.get("prefix", "GH")
    sequence = db.query(OrderSequence).filter(OrderSequence.prefix == prefix).with_for_update().first()
    if not sequence:
        sequence = OrderSequence(prefix=prefix, last_number=99)
        db.add(sequence)
    sequence.last_number += 1
    db.commit()
    order_number = f"{prefix}{sequence.last_number}"
    order_dict = {**order_data, "order_number": order_number, "created_at": datetime.now(), "updated_at": datetime.now()}
    new_order = Order(**order_dict)
    db.add(new_order)
    db.commit()
    db.refresh(new_order)
    return new_order.to_dict()

@router.get("/orders/{order_id}")
def get_order(order_id: int, db: Session = Depends(get_db)):
    order = db.query(Order).filter(Order.id == order_id).first()
    if not order:
        raise HTTPException(status_code=404, detail="Order not found")
    return order.to_dict()

@router.patch("/orders/{order_id}")
def update_order(order_id: int, order_data: dict, db: Session = Depends(get_db)):
    order = db.query(Order).filter(Order.id == order_id).first()
    if not order:
        raise HTTPException(status_code=404, detail="Order not found")
    for key, value in order_data.items():
        setattr(order, key, value)
    db.commit()
    db.refresh(order)
    return order.to_dict()

@router.delete("/orders/{order_id}")
def delete_order(order_id: int, db: Session = Depends(get_db)):
    order = db.query(Order).filter(Order.id == order_id).first()
    if not order:
        raise HTTPException(status_code=404, detail="Order not found")
    db.delete(order)
    db.commit()
    return {"status": "deleted"}

class Order(Base):
    __tablename__ = "orders"

    id = Column(Integer, primary_key=True, index=True)
    customer_id = Column(Integer, ForeignKey("users.id"))
    status = Column(String(50))
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # JSON fields
    cleaning_details = Column(JSON, nullable=True)
    measurements = Column(JSON, nullable=True)
    notes = Column(JSON, nullable=True)
    
    # Relationships
    customer = relationship("User", back_populates="orders")
    
    def __init__(self, **kwargs):
        # Convert JSON strings to Python objects if needed
        for field in ['cleaning_details', 'measurements', 'notes']:
            if field in kwargs and isinstance(kwargs[field], str):
                try:
                    kwargs[field] = json.loads(kwargs[field])
                except json.JSONDecodeError:
                    kwargs[field] = {}
        super().__init__(**kwargs)
    
    def to_dict(self):
        """Convert order to dictionary with proper JSON handling"""
        return {
            'id': self.id,
            'customer_id': self.customer_id,
            'status': self.status,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'updated_at': self.updated_at.isoformat() if self.updated_at else None,
            'cleaning_details': self.cleaning_details,
            'measurements': self.measurements,
            'notes': self.notes
        }