# backend/app/database.py
import os
import logging
import pathlib
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker
from sqlalchemy.ext.declarative import declarative_base

# --- THIS IS THE DEFINITIVE FIX ---
# We will explicitly find the project root and load the .env file from there.
# This prevents any ambiguity about where the application is being run from.
from dotenv import load_dotenv

# This path logic assumes your .env file is in the 'backend' folder,
# e.g., F:\Projects\athletic-clean-platform\backend\.env
project_root = pathlib.Path(__file__).parent.parent.resolve()
dotenv_path = project_root / ".env"

if dotenv_path.exists():
    load_dotenv(dotenv_path=dotenv_path)
    print(f"--- SUCCESS: Loaded .env file from: {dotenv_path} ---")
else:
    # This is a critical warning. If you see this, the path is wrong.
    print(f"--- FATAL WARNING: .env file NOT FOUND at expected path: {dotenv_path} ---")
# --- END FIX ---


# Set up logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


# Now, read the environment variables that were just loaded.
DB_HOST = os.getenv("DB_HOST")
DB_USER = os.getenv("DB_USER")
DB_PASS = os.getenv("DB_PASS")
DB_NAME = os.getenv("DB_NAME")

# Construct the MariaDB connection string.
if DB_HOST and DB_USER and DB_PASS and DB_NAME:
    DATABASE_URL = f"mysql+mysqlconnector://{DB_USER}:{DB_PASS}@{DB_HOST}/{DB_NAME}"
    logger.info(f"Attempting to connect to MariaDB database: {DB_NAME} on host {DB_HOST}")
else:
    # This fallback should now only be used if your .env file is missing or empty.
    fallback_path = project_root / "data" / "athleticclean_M.db"
    fallback_path.parent.mkdir(parents=True, exist_ok=True)
    DATABASE_URL = f"sqlite:///{fallback_path}"
    logger.warning(f"DATABASE .env variables not loaded. Falling back to SQLite: {DATABASE_URL}")

# Create SQLAlchemy engine
engine = create_engine(
    DATABASE_URL,
    connect_args={"check_same_thread": False} if DATABASE_URL.startswith("sqlite") else {},
    echo=True, # Logs all SQL statements for debugging.
    pool_pre_ping=True # Verifies connections before use
)

SessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)
Base = declarative_base()

# Dependency to get a DB session in API endpoints
def get_db():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()

# Initialize database (create tables if they don't exist)
def init_db():
    try:
        # Import all models here so Base knows about them
        # This is the section that has been corrected.
        from .models.order import Order
        from .models.customer import Customer
        from .models.store import Store
        from .models.order_blanket import OrderBlanket
        from .models.admin_note import AdminNote
        from .models.order_sequence import OrderSequence
        
        Base.metadata.create_all(bind=engine)
        logger.info("Database tables verified/created successfully.")
    except Exception as e:
        logger.error(f"Error during database initialization: {str(e)}")
        raise
